let dadosExtraidos = null;
let logCount = 0;
let stats = {
    produtos: 0,
    imagens: 0,
    variacoes: 0,
    avaliacoes: 0
};

document.getElementById('btnExtrair').addEventListener('click', extrairDados);
document.getElementById('btnBaixar').addEventListener('click', baixarJSON);
document.getElementById('btnLimparLogs').addEventListener('click', limparLogs);

// Limpa polling quando popup fecha
window.addEventListener('beforeunload', () => {
    pararPolling();
    chrome.storage.local.remove('extraction-update');
});

// Escuta mensagens do content script em tempo real
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'extraction-update') {
        atualizarPainelTempoReal(message);
    }
    return true;
});

// Escuta mudanças no storage para pegar atualizações mesmo se popup não estava aberto
chrome.storage.onChanged.addListener((changes, areaName) => {
    if (areaName === 'local' && changes['extraction-update']) {
        const update = changes['extraction-update'].newValue;
        if (update) {
            atualizarPainelTempoReal({
                updateType: update.updateType,
                message: update.message,
                data: update.data
            });
        }
    }
});

// Polling adicional para garantir que não perca atualizações
let pollingInterval = null;
function iniciarPolling() {
    if (pollingInterval) return; // Já está rodando
    
    pollingInterval = setInterval(async () => {
        try {
            const result = await chrome.storage.local.get('extraction-update');
            if (result['extraction-update']) {
                const update = result['extraction-update'];
                // Só processa se for recente (últimos 5 segundos)
                if (Date.now() - update.timestamp < 5000) {
                    atualizarPainelTempoReal({
                        updateType: update.updateType,
                        message: update.message,
                        data: update.data
                    });
                }
            }
        } catch (err) {
            // Ignora erros
        }
    }, 200); // Verifica a cada 200ms
}

function pararPolling() {
    if (pollingInterval) {
        clearInterval(pollingInterval);
        pollingInterval = null;
    }
}

function mostrarStatus(mensagem, tipo = 'info') {
    const status = document.getElementById('status');
    status.className = `status status-${tipo}`;
    status.innerHTML = mensagem;
    status.style.display = 'block';
}

function mostrarLoading(show) {
    document.getElementById('loading').style.display = show ? 'block' : 'none';
    document.getElementById('btnExtrair').disabled = show;
}

async function extrairDados() {
    // Reset do painel
    limparLogs();
    resetarStats();
    mostrarPainelTempoReal(true);
    mostrarLoading(true);
    mostrarStatus('🔍 Verificando página...', 'info');
    adicionarLog('🔍 Iniciando extração de dados...', 'info');
    
    // Limpa storage anterior e inicia polling
    chrome.storage.local.remove('extraction-update');
    iniciarPolling();
    
    try {
        // Pega a aba atual
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        
        // Verifica se é uma página do TikTok
        if (!tab.url.includes('tiktok.com')) {
            mostrarStatus('❌ Abra uma página do TikTok Shop primeiro!', 'error');
            adicionarLog('❌ Erro: Página do TikTok não encontrada', 'error');
            mostrarLoading(false);
            return;
        }
        
        adicionarLog('✅ Página do TikTok detectada', 'success');
        atualizarProgresso(10);
        
        // Injeta e executa o script de extração
        let results;
        try {
            results = await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            function: extrairDadosDaPagina
        });
        } catch (scriptError) {
            mostrarLoading(false);
            console.error('Erro ao executar script:', scriptError);
            mostrarStatus('❌ Erro ao executar script: ' + scriptError.message, 'error');
            adicionarLog('❌ Erro ao executar script: ' + scriptError.message, 'error');
            return;
        }
        
        mostrarLoading(false);
        
        if (!results || !results[0] || !results[0].result) {
            mostrarStatus('❌ Erro ao extrair dados - resultado vazio', 'error');
            adicionarLog('❌ Erro ao executar script de extração - resultado vazio', 'error');
            return;
        }
        
        const dados = results[0].result;
        
        if (dados.erro) {
            mostrarStatus('❌ ' + dados.erro, 'error');
            adicionarLog('❌ ' + dados.erro, 'error');
            return;
        }
        
        dadosExtraidos = dados;
        atualizarStats(dados);
        atualizarProgresso(100);
        adicionarLog('✅ Extração concluída com sucesso!', 'success');
        mostrarPreview(dados);
        document.getElementById('btnBaixar').style.display = 'block';
        mostrarStatus('✅ Dados extraídos com sucesso!', 'success');
        
        // Para o polling após conclusão
        pararPolling();
        
    } catch (error) {
        mostrarLoading(false);
        console.error('Erro:', error);
        mostrarStatus('❌ Erro: ' + error.message, 'error');
        adicionarLog('❌ Erro: ' + error.message, 'error');
        
        // Para o polling em caso de erro
        pararPolling();
    }
}

// Nota: A função enviarUpdate agora está definida dentro de extrairDadosDaPagina
// porque precisa ser executada no contexto da página web, não no contexto do popup

// Função que será injetada na página
function extrairDadosDaPagina() {
    return new Promise((resolve) => {
        // Timeout de segurança - resolve após 60 segundos
        const timeout = setTimeout(() => {
            resolve({ erro: 'Timeout: A extração demorou mais de 60 segundos e foi cancelada.' });
        }, 60000);
        
        try {
            // Função auxiliar para enviar atualizações (definida no contexto da página)
            function enviarUpdate(tipo, mensagem, dados = {}) {
                try {
                    // Envia mensagem via window.postMessage para o content script
                    window.postMessage({
                        type: 'extraction-update',
                        updateType: tipo,
                        message: mensagem,
                        data: dados
                    }, '*');
                } catch (e) {
                    // Ignora erros de comunicação
                    console.error('Erro ao enviar update:', e);
                }
            }
            
            enviarUpdate('info', '📜 Iniciando rolagem da página...', { progresso: 5 });
            
            // Rola a página para carregar produtos relacionados
            let scrollCount = 0;
            const maxScrolls = 5;
            
            function rolarPagina() {
                if (scrollCount < maxScrolls) {
                    window.scrollTo(0, document.body.scrollHeight);
                    scrollCount++;
                    enviarUpdate('info', `📜 Rolando página (${scrollCount}/${maxScrolls})...`, { progresso: 10 + (scrollCount * 5) });
                    setTimeout(rolarPagina, 1500);
                } else {
                    enviarUpdate('success', '✅ Rolagem concluída', { progresso: 35 });
                    setTimeout(extrair, 2000);
                }
            }
            
            function extrair() {
                enviarUpdate('info', '🔍 Procurando dados do produto...', { progresso: 40 });
                
                const scriptTag = document.querySelector('script#__MODERN_ROUTER_DATA__');
                
                if (!scriptTag) {
                    clearTimeout(timeout);
                    enviarUpdate('error', '❌ Dados não encontrados', { progresso: 0 });
                    resolve({ erro: 'Dados não encontrados. Certifique-se de estar em uma página de produto.' });
                    return;
                }
                
                enviarUpdate('success', '✅ Script de dados encontrado', { progresso: 45 });
                enviarUpdate('info', '📊 Parseando dados JSON...', { progresso: 50 });
                
                let dados;
                try {
                    dados = JSON.parse(scriptTag.textContent);
                } catch (parseError) {
                    clearTimeout(timeout);
                    enviarUpdate('error', '❌ Erro ao fazer parse do JSON: ' + parseError.message, { progresso: 0 });
                    resolve({ erro: 'Erro ao fazer parse dos dados JSON: ' + parseError.message });
                    return;
                }
                const loaderData = dados.loaderData || {};
                const pageData = loaderData['(name$)/(id)/page'] || {};
                const initialData = pageData.initialData || {};
                const produtoData = initialData.productInfo || {};
                
                if (!produtoData.product_id) {
                    clearTimeout(timeout);
                    enviarUpdate('error', '❌ Produto não encontrado', { progresso: 0 });
                    resolve({ erro: 'Produto não encontrado.' });
                    return;
                }
                
                enviarUpdate('success', '✅ Produto encontrado!', { progresso: 55 });
                
                const produtoBase = produtoData.product_base || {};
                const vendedor = produtoData.seller || {};
                const reviewData = produtoData.product_detail_review || {};
                const logistica = produtoData.logistic || {};
                
                enviarUpdate('info', '📝 Extraindo informações básicas...', { progresso: 60 });
                
                let resultado;
                let imagensDescricao = []; // Imagens encontradas na descrição (declarada fora do try para ser acessível no setTimeout)
                try {
                    // Extrai descrição de múltiplas fontes possíveis
                    let descricao = '';
                    
                    // Função auxiliar para processar array de descrição (texto e imagens)
                    function processarDescricaoArray(descArray) {
                        const textos = [];
                        const imagens = [];
                        
                        if (!Array.isArray(descArray)) return { textos: [], imagens: [] };
                        
                        descArray.forEach(item => {
                            if (!item || typeof item !== 'object') return;
                            
                            // Processa textos
                            if (item.type === 'text' && item.text) {
                                textos.push(item.text.trim());
                            }
                            
                            // Processa imagens
                            if (item.type === 'image' && item.image) {
                                const img = item.image;
                                if (img.url_list && Array.isArray(img.url_list) && img.url_list.length > 0) {
                                    imagens.push({
                                        url_original: img.url_list[0] || '',
                                        url_thumb: img.url_list[0] || '',
                                        origem: 'descricao'
                                    });
                                }
                            }
                        });
                        
                        return { textos, imagens };
                    }
                    
                    // 1. Tenta desc_detail (pode ser array, objeto, string JSON ou string simples)
                    if (produtoBase.desc_detail) {
                        if (typeof produtoBase.desc_detail === 'string') {
                            // Tenta fazer parse se for JSON
                            try {
                                const parsed = JSON.parse(produtoBase.desc_detail);
                                if (Array.isArray(parsed)) {
                                    const resultado = processarDescricaoArray(parsed);
                                    descricao = resultado.textos.join('\n');
                                    imagensDescricao = resultado.imagens;
                                } else {
                                    // Se não for array, usa como string simples
                                    descricao = produtoBase.desc_detail;
                                }
                            } catch (e) {
                                // Se não for JSON válido, usa como string simples
                                descricao = produtoBase.desc_detail;
                            }
                        } else if (Array.isArray(produtoBase.desc_detail)) {
                            // Processa array de objetos de descrição
                            const resultado = processarDescricaoArray(produtoBase.desc_detail);
                            descricao = resultado.textos.join('\n');
                            imagensDescricao = resultado.imagens;
                        } else if (typeof produtoBase.desc_detail === 'object') {
                            // Se for objeto, tenta extrair propriedades comuns
                            descricao = produtoBase.desc_detail.text || 
                                       produtoBase.desc_detail.content || 
                                       produtoBase.desc_detail.value || 
                                       produtoBase.desc_detail.description || '';
                        }
                    }
                    
                    // 2. Se ainda estiver vazia, tenta outros campos em produtoBase
                    if (!descricao || descricao.trim().length === 0) {
                        descricao = produtoBase.description || 
                                   produtoBase.desc || 
                                   produtoBase.detail || 
                                   produtoBase.product_description ||
                                   produtoBase.desc_text ||
                                   produtoBase.desc_content || '';
                    }
                    
                    // 3. Se ainda estiver vazia, tenta em produtoData diretamente
                    if (!descricao || descricao.trim().length === 0) {
                        descricao = produtoData.description || 
                                   produtoData.desc || 
                                   produtoData.detail || 
                                   produtoData.product_description || '';
                    }
                    
                    // 4. Se ainda estiver vazia, tenta extrair do DOM como fallback
                    if (!descricao || descricao.trim().length === 0) {
                        try {
                            const descSelectors = [
                                '.product-description',
                                '.product-detail', 
                                '[data-testid*="description"]',
                                '.description',
                                '.product-info-description',
                                '[class*="description"]',
                                '[class*="detail"]'
                            ];
                            
                            for (const selector of descSelectors) {
                                const descElement = document.querySelector(selector);
                                if (descElement) {
                                    const texto = descElement.textContent?.trim() || descElement.innerText?.trim() || '';
                                    if (texto && texto.length > 10) { // Mínimo de 10 caracteres para considerar válido
                                        descricao = texto;
                                        break;
                                    }
                                }
                            }
                        } catch (domErr) {
                            // Ignora erros do DOM
                        }
                    }
                    
                    // Log de debug (apenas no console, não envia update)
                    if (descricao && descricao.trim().length > 0) {
                        console.log('✅ Descrição extraída com sucesso, tamanho:', descricao.length);
                    } else {
                        console.warn('⚠️ Descrição não encontrada. Campos disponíveis:', {
                            desc_detail: produtoBase.desc_detail,
                            description: produtoBase.description,
                            desc: produtoBase.desc,
                            produtoData_keys: Object.keys(produtoData || {}),
                            produtoBase_keys: Object.keys(produtoBase || {})
                        });
                    }
                    
                    // Garante que specifications seja um array
                    const specifications = Array.isArray(produtoBase.specifications)
                        ? produtoBase.specifications
                        : (produtoBase.specifications ? [produtoBase.specifications] : []);
                    
                    resultado = {
                    sucesso: true,
                    produto: {
                        id: produtoData.product_id,
                            nome: produtoBase.title || '',
                            descricao: descricao || '',
                        vendidos: produtoBase.sold_count || 0,
                            especificacoes: specifications
                    },
                    vendedor: {
                            id: vendedor.seller_id || '',
                            nome: vendedor.name || '',
                            localizacao: vendedor.seller_location || '',
                        total_produtos: vendedor.product_count || 0
                    },
                        imagens: [],
                    avaliacoes: {
                        nota_media: reviewData.product_rating || 0,
                        total: reviewData.review_count_str || '0',
                        total_numerico: reviewData.review_count || 0,
                            comentarios: []
                        },
                        variacoes: [],
                    preco: {},
                    frete: {
                        metodo: logistica.delivery_name || '',
                        prazo: logistica.lead_time || '',
                        valor: logistica.shipping_fee?.price_str || '',
                        gratis: logistica.free_shipping || false,
                        dias_min: logistica.delivery_min_days || 0,
                        dias_max: logistica.delivery_max_days || 0
                    },
                    produtos_relacionados: []
                };
                    enviarUpdate('success', '✅ Informações básicas extraídas', { progresso: 62 });
                } catch (err) {
                    clearTimeout(timeout);
                    enviarUpdate('error', '❌ Erro ao extrair informações básicas: ' + err.message, { progresso: 0 });
                    resolve({ erro: 'Erro ao extrair informações básicas: ' + err.message });
                    return;
                }
                
                // Usa setTimeout para processar de forma assíncrona e não travar
                setTimeout(() => {
                    // Extrai imagens
                    try {
                        enviarUpdate('info', '🖼️ Extraindo imagens...', { progresso: 65 });
                        const images = Array.isArray(produtoBase.images) ? produtoBase.images : [];
                        resultado.imagens = images.map(img => ({
                            url_original: img?.url_list?.[0] || '',
                            url_thumb: img?.thumb_url_list?.[0] || '',
                            origem: 'galeria'
                        })).filter(img => img.url_original);
                        
                        // Adiciona imagens da descrição (se houver)
                        if (imagensDescricao && imagensDescricao.length > 0) {
                            resultado.imagens = resultado.imagens.concat(imagensDescricao);
                            enviarUpdate('info', `📸 ${imagensDescricao.length} imagem(ns) da descrição adicionada(s)`, { progresso: 67 });
                        }
                        
                        enviarUpdate('success', `✅ ${resultado.imagens.length} imagem(ns) extraída(s)`, { 
                            progresso: 68,
                            imagens: resultado.imagens.length 
                        });
                    } catch (err) {
                        enviarUpdate('warning', '⚠️ Erro ao extrair imagens: ' + err.message, { progresso: 68 });
                        resultado.imagens = [];
                    }
                    
                    setTimeout(() => {
                        // Extrai avaliações
                        try {
                            enviarUpdate('info', '⭐ Extraindo avaliações...', { progresso: 70 });
                            const reviewItems = Array.isArray(reviewData.review_items) ? reviewData.review_items : [];
                            resultado.avaliacoes.comentarios = reviewItems.slice(0, 20).map(item => {
                                const reviewImages = Array.isArray(item?.review?.images) ? item.review.images : [];
                                return {
                                    id: item?.review?.review_id || '',
                                    nota: item?.review?.rating || 0,
                                    texto: item?.review?.display_text || '',
                                    data: item?.review?.review_timestamp_fmt || '',
                                    usuario: item?.review_user?.name || 'Anônimo',
                                    sku: item?.sku_specification || '',
                                    imagens: reviewImages.map(img => img?.url_list?.[0]).filter(Boolean)
                                };
                            });
                            enviarUpdate('success', `✅ ${resultado.avaliacoes.comentarios.length} avaliação(ões) extraída(s)`, { 
                                progresso: 75,
                                avaliacoes: resultado.avaliacoes.comentarios.length 
                            });
                        } catch (err) {
                            enviarUpdate('warning', '⚠️ Erro ao extrair avaliações: ' + err.message, { progresso: 75 });
                            resultado.avaliacoes.comentarios = [];
                        }
                        
                        setTimeout(() => {
                            // Extrai variações
                            try {
                                enviarUpdate('info', '🎨 Extraindo variações (SKUs)...', { progresso: 78 });
                                const skus = Array.isArray(produtoData.skus) ? produtoData.skus : [];
                                resultado.variacoes = skus.map(sku => ({
                                    sku_id: sku?.sku_id || '',
                                    preco: sku?.price?.real_price?.price_val || 0,
                                    preco_formatado: sku?.price?.real_price?.price_str || '',
                                    moeda: sku?.price?.real_price?.currency || 'BRL',
                                    estoque: sku?.stock || 0,
                                    disponivel: sku?.status === 1
                                }));
                                enviarUpdate('success', `✅ ${resultado.variacoes.length} variação(ões) extraída(s)`, { 
                                    progresso: 82,
                                    variacoes: resultado.variacoes.length 
                                });
                            } catch (err) {
                                enviarUpdate('warning', '⚠️ Erro ao extrair variações: ' + err.message, { progresso: 82 });
                                resultado.variacoes = [];
                            }
                            
                            setTimeout(() => {
                // Calcula faixa de preço
                                try {
                                    enviarUpdate('info', '💰 Calculando faixa de preços...', { progresso: 85 });
                const precos = resultado.variacoes
                                        .map(v => parseFloat(v.preco) || 0)
                                        .filter(p => !isNaN(p) && p > 0);
                
                if (precos.length > 0) {
                    resultado.preco = {
                        minimo: Math.min(...precos),
                        maximo: Math.max(...precos),
                                            moeda: resultado.variacoes[0]?.moeda || 'BRL'
                    };
                                    }
                                } catch (err) {
                                    enviarUpdate('warning', '⚠️ Erro ao calcular preços: ' + err.message, { progresso: 85 });
                }
                
                                setTimeout(() => {
                // Extrai produtos relacionados
                                    try {
                                        enviarUpdate('info', '🔗 Procurando produtos relacionados...', { progresso: 88 });
                                        
                                        // Procura por diferentes containers possíveis
                const feedList = document.querySelector('.feed-list-wrapper');
                const productsModule = document.querySelector('.products-module');
                                        const productList = document.querySelector('.product-list-dLhdT1');
                                        
                                        const containers = [feedList, productsModule, productList].filter(Boolean);
                                        
                                        // Se não encontrou containers específicos, procura por qualquer elemento com produtos
                                        if (containers.length === 0) {
                                            const allProductWrappers = document.querySelectorAll('.product-wrapper');
                                            if (allProductWrappers.length > 0) {
                                                containers.push(document.body);
                                            }
                                        }
                                        
                                        containers.forEach(container => {
                    if (!container) return;
                    
                                            try {
                                                // Procura por links de produtos - múltiplos seletores
                                                const links = container.querySelectorAll('a[href*="/pdp/"], a[href*="/product/"], a[id="product_card_title_a"]');
                    
                    links.forEach(link => {
                                                    try {
                        const href = link.getAttribute('href');
                                                        if (!href || href === '#') return;
                                                        
                                                        // Encontra o wrapper do produto
                                                        const productWrapper = link.closest('.product-wrapper') || link.closest('.mb-8') || link.closest('div');
                                                        if (!productWrapper) return;
                                                        
                                                        // Extrai imagem
                                                        const img = productWrapper.querySelector('img.product-image, img.product-image-lazy, img[data-src]') || link.querySelector('img');
                                                        const imgSrc = img?.src || img?.getAttribute('data-src') || img?.dataset?.src || '';
                                                        
                                                        // Extrai nome do produto
                                                        const titleElement = productWrapper.querySelector('h3.product-title, .product-title, a[id="product_card_title_a"]') || link;
                                                        const nome = titleElement?.textContent?.trim() || img?.alt || link.textContent?.trim() || '';
                                                        
                                                        // Extrai preço
                                                        let preco = '';
                                                        const priceContainer = productWrapper.querySelector('.product-price');
                                                        if (priceContainer) {
                                                            const salePrice = priceContainer.querySelector('.sale-price');
                                                            if (salePrice) {
                                                                const symbol = salePrice.querySelector('.symbol')?.textContent?.trim() || '';
                                                                const price = salePrice.querySelector('.price')?.textContent?.trim() || '';
                                                                preco = (symbol + ' ' + price).trim();
                                                            }
                                                        }
                                                        
                                                        // Se não encontrou preço, tenta regex no texto
                                                        if (!preco) {
                                                            const texto = productWrapper.textContent || '';
                                                            const match = texto.match(/R\$\s*[\d,.]+/);
                                                            if (match) preco = match[0];
                                                        }
                                                        
                                                        // Monta URL completa
                                                        let url = href;
                                                        if (!url.startsWith('http')) {
                                                            url = url.startsWith('/') ? 'https://www.tiktok.com' + url : 'https://www.tiktok.com/' + url;
                                                        }
                        
                        const produto = {
                                                            url: url,
                                                            nome: nome.substring(0, 200),
                                                            imagem: imgSrc,
                                                            preco: preco
                                                        };
                                                        
                                                        // Verifica se já não existe e se tem dados válidos
                                                        if (produto.url && 
                                                            produto.url !== 'https://www.tiktok.com' && 
                                                            produto.url.includes('/pdp/') &&
                                                            !resultado.produtos_relacionados.find(p => p.url === produto.url)) {
                            resultado.produtos_relacionados.push(produto);
                                                        }
                                                    } catch (linkErr) {
                                                        // Ignora erros em links individuais
                                                        console.debug('Erro ao processar link:', linkErr);
                                                    }
                                                });
                                            } catch (containerErr) {
                                                // Ignora erros em containers
                                                console.debug('Erro ao processar container:', containerErr);
                                            }
                                        });
                                        
                                        enviarUpdate('success', `✅ ${resultado.produtos_relacionados.length} produto(s) relacionado(s) encontrado(s)`, { 
                                            progresso: 95,
                                            produtos: resultado.produtos_relacionados.length 
                                        });
                                    } catch (err) {
                                        enviarUpdate('warning', '⚠️ Erro ao extrair produtos relacionados: ' + err.message, { progresso: 95 });
                                        resultado.produtos_relacionados = [];
                                    }
                                    
                                    enviarUpdate('success', '🎉 Extração finalizada!', { progresso: 100 });
                                    clearTimeout(timeout);
                resolve(resultado);
                                }, 100);
                            }, 100);
                        }, 100);
                    }, 100);
                }, 100);
            }
            
            // Inicia rolagem
            rolarPagina();
            
        } catch (error) {
            clearTimeout(timeout);
            enviarUpdate('error', '❌ Erro: ' + error.message, { progresso: 0 });
            resolve({ erro: error.message });
        }
    });
}

function mostrarPreview(dados) {
    const preview = document.getElementById('preview');
    
    let html = `
        <div class="preview-item">
            <strong>Produto:</strong> ${dados.produto.nome}
        </div>
        <div class="preview-item">
            <strong>Vendedor:</strong> ${dados.vendedor.nome}
        </div>
        <div class="preview-item">
            <strong>Vendidos:</strong> ${dados.produto.vendidos}
        </div>
        <div class="preview-item">
            <strong>Imagens:</strong> ${dados.imagens.length}
        </div>
        <div class="preview-item">
            <strong>Variações:</strong> ${dados.variacoes.length}
        </div>
        <div class="preview-item">
            <strong>Avaliações:</strong> ${dados.avaliacoes.nota_media}/5.0 (${dados.avaliacoes.total})
        </div>
        <div class="preview-item">
            <strong>Produtos Relacionados:</strong> ${dados.produtos_relacionados.length}
        </div>
    `;
    
    preview.innerHTML = html;
    preview.style.display = 'block';
}

function atualizarPainelTempoReal(message) {
    if (message.updateType) {
        adicionarLog(message.message, message.updateType);
    }
    
    if (message.data) {
        if (message.data.progresso !== undefined) {
            atualizarProgresso(message.data.progresso);
        }
        if (message.data.imagens !== undefined) {
            atualizarStat('imagens', message.data.imagens);
        }
        if (message.data.avaliacoes !== undefined) {
            atualizarStat('avaliacoes', message.data.avaliacoes);
        }
        if (message.data.variacoes !== undefined) {
            atualizarStat('variacoes', message.data.variacoes);
        }
        if (message.data.produtos !== undefined) {
            atualizarStat('produtos', message.data.produtos);
        }
    }
}

function adicionarLog(mensagem, tipo = 'info') {
    const logContainer = document.getElementById('logContainer');
    const logItem = document.createElement('div');
    logItem.className = `log-item ${tipo}`;
    
    const time = new Date().toLocaleTimeString('pt-BR');
    const icons = {
        'info': 'ℹ️',
        'success': '✅',
        'error': '❌',
        'warning': '⚠️'
    };
    
    logItem.innerHTML = `
        <span class="log-time">${time}</span>
        <span class="log-icon">${icons[tipo] || '📝'}</span>
        <span>${mensagem}</span>
    `;
    
    logContainer.insertBefore(logItem, logContainer.firstChild);
    logCount++;
    
    // Limita a 50 logs
    if (logCount > 50) {
        const logs = logContainer.querySelectorAll('.log-item');
        if (logs.length > 50) {
            logs[logs.length - 1].remove();
        }
    }
}

function atualizarProgresso(percentual) {
    const progressFill = document.getElementById('progressFill');
    progressFill.style.width = percentual + '%';
}

function atualizarStat(tipo, valor) {
    const statElement = document.getElementById(`stat${tipo.charAt(0).toUpperCase() + tipo.slice(1)}`);
    if (statElement) {
        statElement.textContent = valor;
        stats[tipo] = valor;
    }
}

function resetarStats() {
    stats = {
        produtos: 0,
        imagens: 0,
        variacoes: 0,
        avaliacoes: 0
    };
    atualizarStat('produtos', 0);
    atualizarStat('imagens', 0);
    atualizarStat('variacoes', 0);
    atualizarStat('avaliacoes', 0);
    atualizarProgresso(0);
}

function atualizarStats(dados) {
    atualizarStat('produtos', dados.produtos_relacionados.length + 1);
    atualizarStat('imagens', dados.imagens.length);
    atualizarStat('variacoes', dados.variacoes.length);
    atualizarStat('avaliacoes', dados.avaliacoes.comentarios.length);
}

function mostrarPainelTempoReal(show) {
    const painel = document.getElementById('painelTempoReal');
    if (show) {
        painel.classList.add('active');
    } else {
        painel.classList.remove('active');
    }
}

function limparLogs() {
    const logContainer = document.getElementById('logContainer');
    logContainer.innerHTML = '';
    logCount = 0;
}

function baixarJSON() {
    if (!dadosExtraidos) {
        mostrarStatus('❌ Extraia os dados primeiro!', 'error');
        return;
    }
    
    const blob = new Blob([JSON.stringify(dadosExtraidos, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    
    chrome.downloads.download({
        url: url,
        filename: `tiktok_produto_${dadosExtraidos.produto.id}.json`,
        saveAs: true
    }, () => {
        mostrarStatus('✅ JSON baixado!', 'success');
        setTimeout(() => URL.revokeObjectURL(url), 1000);
    });
}
